<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package direct-project-innovation-initiative
* @subpackage models
*/

/** require parent */
require_once 'mailbox.php';


/**
* @package direct-project-innovation-initiative
* @subpackage models
*/
class User extends Mailbox {
	public static $primary_key = 'user_id';
	
	protected $_ldap_entry;
	
	function cn(){
		return element('cn', $this->ldap_entry);
	}
	
	//currently a wrapper for cn - in part to make it clearer to the non-ldap-savvy what this is, but also in case
	//we ever want display name to be a modified version of the cn
	function display_name(){
		return $this->cn;
	}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// ATTACHMENT CACHE FUNCTIONALITY
// Before they're attached to a message, attachments are stored in a cache specific to the logged-in user.  These functions deal with
// parsing/managing the attachments in the cache.
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////	
	
	
	function attachment_cache(){
		return APPPATH.'cache/'. $this->user_name.'/';
	}
	
	function attachments_from_cache(){
		if(!file_exists($this->attachment_cache())) return array();
		$filenames = get_filenames($this->attachment_cache());
		if(!is_array($filenames)) return $this->error->warning('Could not read '.$this->attachment_cache());
		
		natcasesort($filenames);

		$files = array();
		foreach($filenames as $filename){
			$path = $this->attachment_cache().$filename;
			$binary_string = file_get_contents($path);
			if($binary_string === FALSE)
				$this->error->warning('Could not read '.$path);
			else{
				$files[$filename] = $binary_string;
			}
		}
		return $files;
	}
	
	function attachment_cache_size_in_bytes(){
		$attachment_files = $this->attachments_from_cache();
		if(empty($attachment_files)) return 0;
		
		return array_sum(array_map('string_length_in_bytes', $attachment_files));
	}
	
	function file_name_exists_in_attachment_cache($file_name){
		if(!directory_exists($this->attachment_cache())) return false;
		return file_exists($this->attachment_cache().$file_name);
	}
	
	function attachment_exists_in_cache($name, $binary_string){
		if(!array_key_exists($name, $this->attachments_from_cache())) return false;
	
		$current_binary_string = file_get_contents( $this->attachment_cache().$name );
		if($current_binary_string === FALSE){
			$this->error->warning('Could not read '.$this->attachment_cache().$name);
			return true; //assume that they're the same, since we can't determine otherwise
		}
		
		return ($current_binary_string == $binary_string);
	}
	
#TODO - WHAT IF THE USER HAS TWO DIFFERENT COMPOSE WINDOWS OPEN IN TWO DIFFERENT TABS?  WE NEED TO CACHE BY MESSAGE, NOT JUST BY USER
	function add_attachment_to_cache($name, $binary_string){			
		if(!$this->is->nonempty_string($name)) return $this->error->should_be_a_nonempty_string($name);
		if(!is_string($binary_string)) return $this->error->should_be_a_binary_string($binary_string);
		
		//first, make sure we have a cache 
		if(!directory_exists($this->attachment_cache())) mkdir($this->attachment_cache()); //attempt to create the cache if needed
		if(!directory_exists($this->attachment_cache())) return $this->error->warning('Could not create an attachment cache for '.$this->describe().' at '.$this->attachment_cache());
		
		if($this->attachment_exists_in_cache($name, $binary_string)) return true; //attachment is already in cache; this is the EXACT attachment, with the same binary string & everything
		
		
#TODO - MAKE THE FILENAME UNIQUE IF NEEDED
		$path = $this->attachment_cache().$name;		
		$success = file_put_contents($path, $binary_string);	
		if($success === FALSE)
			return $this->error->warning('Unable to write '.$path.' to attachment cache');
		return true;
	}
	
	function clear_attachment_cache(){
		if(!directory_exists($this->attachment_cache())) return true; //we're done!
		delete_files($this->attachment_cache());
		return rmdir($this->attachment_cache());
	}
	
	
////////////////////////////
// GETTERS
////////////////////////////	

	//get the info about this user from LDAP *once* - to force this info to refresh, set $_ldap_entry to null
	public function ldap_entry(){
		if(!isset($this->ldap_entry)){
			$this->_ldap_entry = array(); //by default, set an empty array so that we don't keep trying to access LDAP if the call isn't working
			$CI = get_instance();
			$CI->load->library('ldap');
			$result = $CI->ldap->search(null,1,array('cn'),'(uid='.$this->username.')');
			
			//make sure that the result is an array
			if(!is_array($result)){
				$this->error->warning('No LDAP entries were found for user '.$this->error->describe($this->username));				
			}else{
				//let the developer know something's going wrong if we have more than one LDAP entry for this user
				if(count($result) > 1){
					$this->error->warning(count($result).' LDAP entries were found for user '.$this->error->describe($this->username).' when only one was expected.');
				}
				
				$this->_ldap_entry = first_element($result);
			}
			
		}
		return $this->_ldap_entry;
	}

	public function username(){
		return $this->user_name;
	}
	
/////////////////////////////
// STATIC METHODS
/////////////////////////////	
	
	public static function find_from_session(){
		$username = get_instance()->session->userdata('username');
		if(empty($username)) return false;
		
		$CI = get_instance();
		if(isset($CI->user) && User::is_an_entity($CI->user) && $CI->user->username == $username)
			return $CI->user; //skip the db query
		
		return User::find_one( array('user_name' => $username) );
	}	
	
}
